/*  Durchsucht die Pfade (edges) nach Möglichkeiten die gegebene ABK in dem vorgegebenen Zeitfenster zu terminieren.

    Dabei wird nach freien Lücken auf den möglichen Knoten (vertex) bzw. Kostenstellen gesucht, den jeweiligen AG einzuterminieren.

    Die Endergebnisse landen in der Tabelle "scheduling.resource_termination_results" und es wird die ID dieses Suchlaufs zurückgegeben.

    Die Konstante "_check_depth" begrenzt die Suchtiefe.
    Es werden zum vorhergehenden AG jeweils nur die $_check_depth schnellsten Pfade betrachtet.

    Die Konstante _time_penalty beeinflusst die Reihnfolge der zu betrachtenden Arbeistplätze.
    Pro weiteren noch nicht vorher genutzen Arbeitsplatzressource in aktuellen Pfad wird diese Zeitstrafe zum aktuellen Endtermin dazugerechnet.
    Pfade mit den wenigsten unterschiedlichen Arbeitspaltzressourcen, also möglichst immer die gleichen Kostenstellen werden bevorzugt.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__termination_search', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__termination_search(
      _abk_ix                     integer,            -- ID der zu terminierenden ABK.
      _timeframe_start            timestamp,          -- Minimum des Zeitbereichs, in dem die ABK einterminiert werden soll. Passt die ABK nicht in den Zeitbereich, schlägt Terminierung mit Exception fehl.
      _timeframe_end              timestamp,          -- Maximum des Zeitbereichs, in dem die ABK einterminiert werden soll. Passt die ABK nicht in den Zeitbereich, schlägt Terminierung mit Exception fehl.
      _direction                  varchar = 'forward',  -- Terminbierungsrichtung: Erlaubt sind "forward" (Vorwärtsterminuerung) und "backward" (Rückwärtsterminierung).
      _scenario                   varchar = null,       -- TODO AXS: Unbenutzt! Kann das weg?
      _checkBlockedTimes          bool = true,          -- Bei 'false' DLZ-Terminierung: Ignoriert Blockierungen auf der Ressource durch andere Task*-Einträge. Es werden nur Off-Times beachtet.
      _blocktime_refcursor        refcursor = null,     -- Möglichkeit um von außen via einem refcursor zusätzliche Timeline-Einträge in die Funktion reinzugeben.
      _termination_start_time     timestamp = null,     -- Startzeit des Mittelpunkt-AG -- TODO AXS: Kann vermutlich weg. Mittelpunktterminierung soll von außerhalb erfolgen.
      _termination_start_ab2_id   int = null,           -- Mittelpunkt-AG -- TODO AXS: Kann vermutlich weg. Mittelpunktterminierung soll von außerhalb erfolgen.
      _allow_overlap              bool = false,         -- Überlappungen bei der Terminierung von Arbeitsgängen innerhalb der gleichen ABK erlauben.
      _timeframe_adjusted         bool = false,         -- Bei dem übergebenen Zeitbereich handelt sich um ein angepassten Zeitbereich. Mit dem urspünglichen Zeitbereich könnte der AG-Bereich mit Vorgänger- oder Nachfolger-AGen der gleichen ABK überlappen.
      _allow_fragmentation        varchar = 'no',       -- Fragmeniering erlauben: 'no' - keine Fragmentierung erlauben (Standard); 'until' - Terminierung bis zur ersten Fragmentierung durchführen; 'yes' - Fragmentierung erlauben, ab Fragmentierung werden nur noch Blockierungen gebildet.
      _loglevel                   int DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS int AS $$

  DECLARE
      _prefix  varchar := format( 'resource_timeline__termination_search abk_ix:%L -', _abk_ix );

      _check_depth CONSTANT int       := 3;                 -- Maximale Suchtiefe. Anzahl der maximal zu betrachtenden Arbeitsplätze pro AG.
      _time_penalty CONSTANT interval := interval '3 days'; -- Zeitstrafe pro Hinzufügen einer weiteren noch nicht vorher genutzten Arbeitsplatzressource im aktuellen Ressourcenpfad.

      _ab_ap_nr varchar := ab_ap_nr FROM abk WHERE ab_ix = _abk_ix;

      _ab2_first ab2  :=  ab2 FROM scheduling.vertex t        -- Erster AG im vorgegebenen AG-Bereich (Vertexpfad). Nur für AGe im AG-Bereich wird der Vertex gebildet.
                             JOIN ab2 ON t.ab2_a2_id = a2_id
                             JOIN abk ON a2_ab_ix = ab_ix
                            WHERE a2_ab_ix = _abk_ix
                            ORDER BY a2_n
                              ASC LIMIT 1;

      _ab2_last ab2   :=  ab2 FROM scheduling.vertex t        -- Letzer AG im vorgegebenen AG-Bereich (Vertexpfad). Nur für AGe im AG-Bereich wird der Vertex gebildet.
                             JOIN ab2 ON t.ab2_a2_id = a2_id
                             JOIN abk ON a2_ab_ix = ab_ix
                            WHERE a2_ab_ix = _abk_ix
                            ORDER BY a2_n
                             DESC LIMIT 1;

      _search_id    int := nextval('scheduling.resource_termination_sequence');

      _current_ab2 ab2;
      _next_ab2 ab2;

      _current_start timestamp := _timeframe_start;
      _current_end   timestamp := _timeframe_end;

      _last_step record;

      _last_step_ids int[];
      _visited_steps int[];

      _abk_done bool := false;
      _ab2_done bool := false;

      _safe_guard timestamp := clock_timestamp();
      _timer timestamp := clock_timestamp();

      _turned_around boolean = false;

      _lgz int := 0; -- Liegezeit

      _ab2_prev record; -- Direkter Vorgänger-AG vom ersten zu terminierenden AG in dieser Suche
      _ab2_next record; -- Direkter Nachfolger-AG vom letzten zu terminierenden AG in dieser Suche

  BEGIN

      RAISE NOTICE '%', format(
          $call$
                    call: scheduling.resource_timeline__termination_search( 
                          _abk_ix => %L, 
                          _timeframe_start => %L, 
                          _timeframe_end => %L, 
                          _direction => %L, 
                          _scenario => %L, 
                          _checkBlockedTimes => %L, 
                          _blocktime_refcursor => %L, 
                          _termination_start_time => %L, 
                          _termination_start_ab2_id => %L, 
                          _allow_overlap => %L, 
                          _timeframe_adjusted => %L, 
                          _allow_fragmentation => %L, 
                          _loglevel => %L 
                    )

                    timstamp=%L
          $call$ 
                          ,
                          _abk_ix, 
                          _timeframe_start, 
                          _timeframe_end, 
                          _direction, 
                          _scenario, 
                          _checkBlockedTimes, 
                          _blocktime_refcursor, 
                          _termination_start_time, 
                          _termination_start_ab2_id, 
                          _allow_overlap, 
                          _timeframe_adjusted, 
                          _allow_fragmentation, 
                          _loglevel
                          ,
                          clock_timestamp()::time
      );

      -- Debug
      IF _loglevel >= 5 THEN
        RAISE NOTICE 'searchid %', _search_id;
      END IF;

      IF _ab2_first.a2_id IS null AND _ab2_last.a2_id IS null THEN
          RAISE EXCEPTION 'scheduling.vertex seems null. call scheduling.ab2__resource_requirements_options__ksvba__from__ab2__create for abk: %', _abk_ix;
      END IF;

      -- Terminierter Vorgänger-AG vom ersten zu terminierenden AG in dieser Suche bestimmen.
      IF _ab2_first.a2_id IS NOT null THEN
              SELECT a2_id, a2_ab_ix, a2_n, ti_date_end AS ti_date_end_max, a2_lgz_sek, ti_resource_id
                INTO _ab2_prev
                FROM ab2
                LEFT JOIN scheduling.resource_timeline ON ti_a2_id = a2_id
                                                      AND ti_type IN ( 'task', 'task.buffer' )
               WHERE a2_id = ( tabk.ab2__prior__get( _ab2_first, _filter_out__a2_ende__true => true, _ks_plan_only => true ) ).a2_id -- nur offene AGs (nicht beendet => a2_ende = false)
               ORDER BY ti_date_end DESC
               LIMIT 1;
      END IF;
      -- Terminierter Nachfolger-AG vom letzten zu terminierenden AG in dieser Suche bestimmen.
      IF _ab2_last.a2_id IS NOT null THEN
              SELECT a2_id, a2_ab_ix, a2_n, ti_date_start AS ti_date_start_min, a2_lgz_sek, ti_resource_id
              INTO _ab2_next
              FROM ab2
              LEFT JOIN scheduling.resource_timeline ON ti_a2_id = a2_id
                                                    AND ti_type IN ( 'task', 'task.buffer' )
              WHERE a2_id = ( tabk.ab2__next__get( _ab2_last, _filter_out__a2_ende__true => true, _ks_plan_only => true ) ).a2_id
              ORDER BY ti_date_start ASC
              LIMIT 1;
      END IF;

      -- Ersten bzw. letzten AG als aktuellen AG festlegen.
      IF ( _direction = 'forward' ) THEN
          _current_ab2 := _ab2_first;
      ELSE
          _current_ab2 := _ab2_last;
      END IF;

      RAISE NOTICE 'start ab2 id:% ix:% a2n:, %', _current_ab2.a2_id, _current_ab2.a2_ab_ix, _current_ab2.a2_n;

      -- Überschreibe den Start-AG, wenn ein Mittelpunkt-AG gegeben ist.
      IF ( _termination_start_ab2_id IS NOT null ) THEN
          _current_ab2 := ab2 FROM ab2 WHERE a2_id = _termination_start_ab2_id;
      END IF;

      -- Überschreibe die Startzeit, wenn eine Startzeit für den Mittelpunkt-AG gegeben ist.
      IF ( _termination_start_time IS NOT NULL ) THEN
          _current_start := _termination_start_time;
      END IF;

      -- fake empty struct
      SELECT
        null::int as id,
        null::int as search_id,
        null::int as abk_ab_ix,
        array[]::int[] as resource_id_path,
        array[]::int[] as resource_vertices_path,
        array[]::int[] as ab2_a2_id_path,
        _current_start as date_start,
        _current_end   as date_end,
        array[]::jsonb[] as ab2_termination_result
      INTO _last_step;

      -- Durch die einzelnen AG der ABK durchlaufen.
      << abkloop >>
      LOOP

          -- Den abkloop Beenden, wenn alle gewählten AG der ABK durchlaufen wurden.
          EXIT WHEN _abk_done;

          -- Debug
          IF _loglevel >= 5 THEN
              RAISE NOTICE '[%] working time - current ab2: %, %', clock_timestamp() - _safe_guard, _current_ab2.a2_id,_current_ab2.a2_n;
          END IF;

          IF ( _direction = 'forward' ) THEN
              _next_ab2 :=
                  ab2 FROM scheduling.vertex t
                  JOIN ab2 ON t.ab2_a2_id = a2_id
                  WHERE
                        a2_n > _current_ab2.a2_n
                    AND a2_ab_ix = _abk_ix
                  ORDER BY a2_n ASC LIMIT 1;
          ELSE
              _next_ab2 :=
                  ab2 FROM scheduling.vertex t
                  JOIN ab2 ON t.ab2_a2_id = a2_id
                  WHERE
                        a2_n < _current_ab2.a2_n
                    AND a2_ab_ix = _abk_ix
                  ORDER BY a2_n DESC LIMIT 1;
          END IF;

            -- Wenn Rückwärts, ...
            IF ( _direction <> 'forward' ) THEN
                -- ... dann Liegezeit vom aktuellen AG für aktuelle Schleife holen.
                _lgz := coalesce( _current_ab2.a2_lgz_sek, 0 )::int;
                -- Debug
                IF _loglevel >= 5 THEN
                    RAISE NOTICE '% lgz - a2_n:%; direction:%; lgz:%;', _prefix, _current_ab2.a2_n, _direction, _lgz;
                END IF;
            END IF;


          -- Durchlaufe nacheinander einzeln alle Pfade (edges) des vorhergehenden AG zum aktuellen AG
          -- und gib dabei die Terminierungsergebnisse für den entsprechenden Pfad zurück.
          --
          -- Über das Array "_visited_steps" wird überwacht, welche Pfade vom vorhergehenden AG schon betrachtet wurden.
          -- In dem Array "_last_step_ids" sind alle IDs der Terminierungen des vorhergehenden AG aufgeführt.
          --
          -- Beim Durchlauf für den ersten AG werden pro möglichen Knoten (Vertex) jeweils eine Zeile zurückgegeben.
          -- Man könnte also einen einzigen imaginären Knoten 0 als Ursprung annehmen.
          LOOP
              -- Die aktuelle Schleife des aktuellen Knotens (der aktuellen Ressource).
              EXIT WHEN _ab2_done;

              -- Wenn es zu lange dauert, brich DB-Funktion ab
              -- IF ( clock_timestamp() - interval '60s' > _safe_guard ) THEN
              --     RAISE NOTICE 'safty suicide';
              --     return 0;
              -- END IF;

              -- raise notice 'ab2: %', clock_timestamp() - _safe_guard;

              IF (
                  SELECT count(*) > 0
                  FROM scheduling.resource_termination_results
                  WHERE search_id = _search_id
              ) THEN

                  SELECT *
                  INTO _last_step
                  FROM scheduling.resource_termination_results
                  WHERE
                        search_id = _search_id
                    AND id = ANY ( _last_step_ids )
                    AND id <> ALL ( _visited_steps  )
                  LIMIT 1;

                  IF _last_step IS NULL THEN
                      _ab2_done := true;
                      CONTINUE;
                  END IF;

              END IF;

              -- raise notice 'current path: %',_last_step.ab2_a2_id_path;
              raise notice 'current ab2: %, %', _current_ab2.a2_n,  _current_ab2.a2_ks;

              -- Die aktuelle ID der möglichen Terminierung des vorhergehenden AG der Liste der bereits betrachteten IDs hinzufügen.
              -- Die Variable "_visited_steps" enthält die Liste breits betrachteten möglichen Terminierung des vorhergehenden AG, somit die Liste der bereits betrachteten Pfade.
              _visited_steps := _visited_steps || _last_step.id;

              -- TODO AXS: Bei Mittelpunktterminierung nach dem Umschalten der Terminierungsrichtung hier Startzeit der Terminerung überschreiben.
              -- TODO AXS: Vermutlich kann Mittelpunktterminierung aus dieser Funktion ganz ausgebaut werden. Soll außerhalb realisiert werden. Dann Parameter "_termination_start_ab2_id" und "_termination_start_time" ausbauen.
              -- push forward times
              IF ( _last_step.id IS NOT NULL ) THEN
                  IF ( _direction = 'forward' ) THEN
                      _current_start := _last_step.date_end;
                      _current_end := _timeframe_end;
                  ELSE
                      _current_start := _timeframe_start;
                      _current_end   := _last_step.date_start;
                  END IF;
              END IF;

              -- check for start/endtime
              -- try catch
              BEGIN

                  INSERT INTO scheduling.resource_termination_results (
                    search_id, abk_ab_ix,
                    resource_id_path, resource_vertices_path, ab2_a2_id_path,
                    date_start, date_end,
                    ab2_termination_result
                  )
                  SELECT
                      _search_id,
                      _abk_ix,

                      -- resource_id_path
                      CASE WHEN _direction = 'forward'
                        THEN  array_append( _last_step.resource_id_path, vert.resource_id )
                        ELSE array_prepend( vert.resource_id, _last_step.resource_id_path )
                      END,

                      -- resource_vertices_path
                      CASE WHEN _direction = 'forward'
                        THEN  array_append( _last_step.resource_vertices_path, vert.id )
                        ELSE array_prepend( vert.id, _last_step.resource_vertices_path )
                      END,

                      -- ab2_path
                      CASE WHEN _direction = 'forward'
                        THEN  array_append( _last_step.ab2_a2_id_path, vert.ab2_a2_id )
                        ELSE array_prepend( vert.ab2_a2_id, _last_step.ab2_a2_id_path )
                      END,

                      -- date_start
                      CASE WHEN _direction = 'forward'
                        THEN  _last_step.date_start
                        ELSE termination_result.slotstartdate
                      END AS date_start,

                      -- date_end
                      CASE WHEN _direction = 'forward'
                        THEN  termination_result.slotenddate
                        ELSE _last_step.date_end
                      END AS date_end,

                      -- ab2_termination_result
                      CASE WHEN _direction = 'forward'
                        THEN  _last_step.ab2_termination_result || to_jsonb(termination_result.raw)
                        ELSE to_jsonb(termination_result.raw) || _last_step.ab2_termination_result
                      END AS ab2_termination_result

                  FROM scheduling.vertex AS vert
                  JOIN ab2 ON vert.ab2_a2_id = a2_id
                  JOIN abk ON ab_ix = a2_ab_ix
                  JOIN LATERAL (
                    SELECT
                      min( slotstartdate ) AS slotstartdate,
                      max( slotenddate ) AS slotenddate,
                      array_agg(calc) AS raw
                    FROM scheduling.resource_timeline__timeslots__search(
                            _resource_id          =>  vert.resource_id ,
                            _time_required        =>  scheduling.ab2__required_worktime__get( ab2.a2_id ),
                            _load_required        =>  vert.load,
                            _timeframe_start      =>  CASE -- TODO AXS: Zur besseren Lesbarkeit die Ermittelung von Wert für _timeframe_start in eigene Funktion packen.
                                                          -- Vorwärts
                                                          WHEN ( _direction = 'forward' )
                                                              THEN
                                                                  CASE
                                                                      -- Wenn erster AG im vorgegebenen AG-Bereich (Vertexpfad) und ...
                                                                      WHEN ( ab2.a2_id = _ab2_first.a2_id )
                                                                          THEN
                                                                              CASE
                                                                                  --  ... es existiert ein terminierter direkter Vorgänger-AG, dann ...
                                                                                  WHEN ( _ab2_prev.a2_id IS NOT null AND _ab2_prev.ti_resource_id IS NOT null AND NOT _allow_overlap )
                                                                                      -- ... Transport- und Liegezeit des Vorgänger-AG beachten und ggf. diese mindestens als Lücke zum aktuellen AG lassen.
                                                                                      THEN greatest( scheduling.resource__transport_time__get( _ab2_prev.ti_date_end_max, _ab2_prev.ti_resource_id, vert.resource_id, 1, _ab2_prev.a2_lgz_sek::int, _loglevel => _loglevel ), _current_start )
                                                                                  -- Wenn ein terminierter direkter Vorgänger-AG fehlt, dann entsteht Lücke aus Transport- und Liegezeit das Vorgänger -AG zum aktuellen AG bei der Terminierung des Vorgänger-AG. Muss hier also nicht beachtet werden.
                                                                                  ELSE _current_start
                                                                              END
                                                                      -- Ansonsten (alle anderen AGe (nicht erster)), dann Transport- und Liegezeit des Vorgänger-AG beachten und ggf. diese mindestens als Lücke zum Vorgänger-AG lassen.
                                                                      ELSE scheduling.resource__transport_time__get( _current_start, _last_step.resource_id_path[ array_length( _last_step.resource_id_path, 1 ) ], vert.resource_id, 1, _lgz, _loglevel => _loglevel )
                                                                  END
                                                          -- Rückwärts
                                                          ELSE
                                                              CASE
                                                                  -- Wenn erster AG und es existiert ein terminierter direkter Vorgänger-AG, dann ...
                                                                  WHEN ( ab2.a2_id = _ab2_first.a2_id AND _ab2_prev.a2_id IS NOT null AND _ab2_prev.ti_resource_id IS NOT null AND NOT _allow_overlap )
                                                                      -- ... Transport- und Liegezeit des Vorgänger-AG beachten und ggf. diese mindestens als Lücke zum aktuellen AG lassen.
                                                                      THEN greatest( scheduling.resource__transport_time__get( _ab2_prev.ti_date_end_max, _ab2_prev.ti_resource_id, vert.resource_id, 1, _ab2_prev.a2_lgz_sek::int, _loglevel => _loglevel ), _current_start )
                                                                  -- Ansosnten (alle anderen AGe (nicht erster oder erster AG ohne terminierten Vorgänger)) entsteht Lücke aus Transport- und Liegezeit das  Vorgänger-AG zum aktuellen AG bei der Terminierung des Vorgänger-AG. Muss hier also nicht beachtet werden.
                                                                  ELSE _current_start
                                                              END
                                                      END, -- start time
                            _timeframe_end        =>  CASE -- TODO AXS: Zur besseren Lesbarkeit die Ermittelung von Wert für _timeframe_end in eigene Funktion packen.
                                                          -- Rückwärts
                                                          WHEN ( _direction <> 'forward' )
                                                              THEN
                                                                  CASE
                                                                      -- Wenn letzter AG im vorgegebenen AG-Bereich (Vertexpfad) und es existiert ein terminierter direkter Nachfolger-AG, dann ...
                                                                      WHEN ( ( ab2.a2_id = _ab2_last.a2_id ) AND ( _ab2_next.a2_id IS NOT null AND _ab2_next.ti_resource_id IS NOT null AND NOT _allow_overlap ) )
                                                                          -- ... Transport- und Liegezeit des aktuellen AG beachten und ggf. diese mindestens als Lücke zum Nachfolger-AG lassen.
                                                                          THEN least( scheduling.resource__transport_time__get( _ab2_next.ti_date_start_min, vert.resource_id, _ab2_next.ti_resource_id, -1, _lgz, _loglevel => _loglevel ), _current_end )
                                                                      -- NOTE AXS: Liegezeitproblem: Erst einmal auskommentiert, damit z.B. Vorwärtsterminierung mit Packen funktioniert. Vorrübergehend direkt in Airbus-Terminierung gelöst.
                                                                      -- -- Wenn letzter terminierbarer AG in der ABK und ein terminierter direkter Nachfolger-AG fehlt, dann ...
                                                                      -- WHEN ( tabk.ab2__next__get( _ab2 => ab2, _ks_plan_only => true ) IS null )
                                                                      --     -- ... Liegezeit zum Maximum des Zeitbereichs beim letzten AG beachten.
                                                                      --     THEN scheduling.resource__transport_time__get( _current_end, vert.resource_id, vert.resource_id, -1, _lgz, _loglevel => _loglevel )
                                                                      -- Wenn letzter AG im vorgegebenen AG-Bereich (Vertexpfad) und ein terminierter direkter Nachfolger-AG fehlt, aber nicht der letzte terminierbare AG in der ABK ist, dann ...
                                                                      WHEN ( ab2.a2_id = _ab2_last.a2_id )
                                                                          --  ... entsteht Lücke aus Transport- und Liegezeit das aktuellen AG zum Nachfolger bei der Terminierung des Nachfolger-AG. Muss hier als nocht beachtet werden.
                                                                          THEN _current_end
                                                                      -- Ansonsten (alle anderen AGe (nicht letzter)), dann Transport- und Liegezeit des aktuellen AG beachten und ggf. diese mindestens als Lücke zum Nachfolger lassen.
                                                                      ELSE scheduling.resource__transport_time__get( _current_end, vert.resource_id, _last_step.resource_id_path[ 1 ] , -1, _lgz, _loglevel => _loglevel )
                                                                   END
                                                          -- Vorwärts
                                                          ELSE
                                                              CASE
                                                                  -- Wenn letzer AG im vorgegebenen AG-Bereich (Vertexpfad) und es existiert ein terminierter direkter Nachfolger-AG, dann ...
                                                                  WHEN ( ( ab2.a2_id = _ab2_last.a2_id ) AND ( _ab2_next.a2_id IS NOT null AND _ab2_next.ti_resource_id IS NOT null AND NOT _allow_overlap ) )
                                                                      -- ... Transport- und Liegezeit des aktuellen AG beachten und ggf. diese mindestens als Lücke zum Nachfolger lassen.
                                                                      THEN least( scheduling.resource__transport_time__get( _ab2_next.ti_date_start_min, vert.resource_id, _ab2_next.ti_resource_id, -1, _lgz, _loglevel => _loglevel ), _current_end )
                                                                  -- NOTE AXS: Liegezeitproblem: Erst einmal auskommentiert, damit z.B. Vorwärtsterminierung mit Packen funktioniert. Vorrübergehend direkt in Airbus-Terminierung gelöst.
                                                                  -- -- Wenn letzer terminierbarer AG in der ABK und ein terminierter direkter Nachfolger-AG fehlt, dann ...
                                                                  -- WHEN ( tabk.ab2__next__get( _ab2 => ab2, _ks_plan_only => true ) IS null )
                                                                  --     -- ... Liegezeit zum Maximum des Zeitbereichs beim letzten AG beachten.
                                                                  --     THEN scheduling.resource__transport_time__get( _current_end, vert.resource_id, vert.resource_id, -1, _lgz, _loglevel => _loglevel )
                                                                  -- Ansonsten (alle anderen AGe (nicht letzter oder letzter AG ohne terminierten Nachfolger)) entsteht Lücke aus Transport- und Liegezeit das aktuellen AG zum Nachfolger-AG bei der Terminierung des Nachfolger-AG. Muss hier also nicht beachtet werden.
                                                                  ELSE _current_end
                                                              END
                                                      END, -- end time
                            _direction            =>  _direction,
                            _allow_fragmentation  =>  _allow_fragmentation,
                            _blocktime_refcursor  =>  _blocktime_refcursor,
                            _checkBlockedTimes    =>  _checkBlockedTimes,
                            _factor_tm_ta         =>  scheduling.ab2__factor_tm_ta__get( ab2.a2_id ),
                            _isBuffer             =>  TSystem.ENUM_ContainsValue( ab_stat, 'BUFFER' ),
                            _loglevel             =>  _loglevel
                         ) AS calc
                  ) termination_result ON true
                  WHERE
                        a2_ab_ix = _abk_ix
                    AND vert.ab2_a2_id = _current_ab2.a2_id
                    AND termination_result.slotEndDate IS NOT NULL
                  ;
              EXCEPTION WHEN OTHERS THEN

                  RAISE EXCEPTION 'Termination failed for abk, ab_ap_nr, a2_n: %, %, %, %', _abk_ix, coalesce( _ab_ap_nr, '(null)' ), _current_ab2.a2_n, sqlerrm;
              END;

              -- Debug
              IF _loglevel >= 5 THEN
                  RAISE NOTICE '% prevr:%, prev:% - first:% ~ current:% ~ last:% - next:%, nextr:%;', _prefix, _ab2_prev.ti_resource_id, _ab2_prev.a2_id, _ab2_first.a2_id, _current_ab2.a2_id, _ab2_last.a2_id, _ab2_next.a2_id, _ab2_next.ti_resource_id;
              END IF;

              IF (
                      -- first pass
                      _last_step_ids IS NULL

                      -- TODO AXS: Nochmal prüfen?
                      -- Doppelt gemoppelt. Diese Prüfung passiert schon am Anfang der Schleife für die Pfade des vorhergehenden AG.
                      -- stored ids from last pass
                   OR _visited_steps @> _last_step_ids AND _visited_steps <@ _last_step_ids
              ) THEN
                  -- raise notice 'done with ab2';
                  -- raise notice 'visited: %',_visited_steps;
                  -- raise notice 'steps:   %', _last_step_ids;

                  _ab2_done := true;
              END IF;

          END LOOP;

          -- raise notice 'ab2 done: %', clock_timestamp() - _safe_guard;

          RAISE NOTICE '_direction=%, _ab2_last.a2_id=%, _next_ab2.a2_id=%', _direction, _ab2_last.a2_id, _next_ab2.a2_id;


          -- Die IDs der möglichen Terminierungen des vorhergehenden AG im Array "_last_step_ids" merken.
          -- Diese IDs werden verwendet bei der Ermittelung der für die Terminierung zu durchlaufen Pfade zum aktuellen AG in der ABK.
          -- Die Anzahl der zu durchlaufenden Pfade wird dabei über die Variable "_check_depth" auf die x schnellsten Pfade beschränkt.
          -- Dabei wird eine Hinzunahme eines neuen Arbeistpaltzes mit einer Zeitstrafe (Variable "_time_penalty") bestraft. Dieser Pfad ist somit vermeintlich um "_time_penalty" langsamer.
          -- Ein Pfad ist der Weg (die Kombination) von einem konkreten möglichen Arbeitsplatz des vorhergehenden AG zu einem konkreten möglichen Arbeitsplatz des aktuellen AG.
          _last_step_ids     :=
            array_agg ( id ) FROM (
                SELECT id
                  FROM scheduling.resource_termination_results
                 WHERE
                       search_id = _search_id
                   AND
                       CASE WHEN _direction = 'forward'
                         THEN ab2_a2_id_path[ array_length( ab2_a2_id_path , 1) ] = _current_ab2.a2_id
                         ELSE ab2_a2_id_path[ 1 ] = _current_ab2.a2_id
                       END
                 ORDER BY
                       -- array_length( array_unique(resource_id_path), 1) ist analog auch im resource_termination_results__abk__fastest_fetch. Beide Stellen sind anzupassen bis ggf klar ist - was das hier macht (siehe TODO über dem Block)
                       -- Sortiert nach schnellster Durchlaufzeit. Dafür werden pro weitere Kostenstelle 3 Tage virtuelle Kosten aufgeschlagen. Damit wird der Pfad der kürzer ist (also auf gleichen Kostenstellen läuft und nicht wechselt) bevorzugt
                       date_end + ( ( array_length( array_unique(resource_id_path), 1) ) * _time_penalty ) ASC,
                       -- bei gleichem Ende-Datum nehmen wir den kürzesten Pfad. Das ist der mit den wenigsten unterschiedlichen Resourcen, also möglichst immer die gleiche Kostenstelle
                       array_length( array_unique(resource_id_path), 1)
                 LIMIT _check_depth
            ) t;

          -- Debug
          IF _loglevel >= 5 THEN
            RAISE NOTICE 'last steps: %', _last_step_ids;
          END IF;

          _visited_steps := array[]::int[];
          _ab2_done := false;

          -- Prüfen, ob der aktuelle AG einterminiert werden konnte.
          /*
          -- TODO muss das extra nochmal abgefragt werden? Was macht der array_agg darüber?
          IF NOT EXISTS (
                            SELECT *
                              FROM scheduling.resource_termination_results
                            WHERE search_id = _search_id
                              AND CASE WHEN _direction = 'forward'
                                        THEN ab2_a2_id_path[ array_length( ab2_a2_id_path , 1) ] = _current_ab2.a2_id
                                        ELSE ab2_a2_id_path[ 1 ] = _current_ab2.a2_id
                                  END
                        ) THEN
          */
          IF tsystem.array_null( _last_step_ids ) THEN
              IF _timeframe_adjusted THEN
                  RAISE EXCEPTION 'Wegen Überlappung mit Vorgänger- bzw. Nachfolger-AG keine Terminierung möglich für abk, ab_ap_nr, a2_n: %, %, %', _abk_ix, coalesce( _ab_ap_nr, '(null)' ), _current_ab2.a2_n;
              END IF;
              RAISE EXCEPTION 'Kein Terminierungsergebnis im vorgegeben Zeitfenster gefunden für abk, ab_ap_nr, a2_n: %, %, % <<"%" - "%">>', _abk_ix, coalesce( _ab_ap_nr, '(null)' ), _current_ab2.a2_n, _timeframe_start, _timeframe_end;
          END IF;

          -- Wenn vorwärts, ...
          IF _direction = 'forward' THEN
              -- ... dann Liegezeit vom aktuellen AG für nächste Schleife holen.
              _lgz := coalesce( _current_ab2.a2_lgz_sek, 0 )::int;
              -- Debug
              IF _loglevel >= 4 THEN
                  RAISE NOTICE '% lgz - a2_n:%; direction:%; lgz:%;', _prefix, _current_ab2.a2_n, _direction, _lgz;
              END IF;
          END IF;

          -- Wir haben alle AGe in die vorgegebene Richtung terminiert und es gibt einen Mittelpunkt-AG ( _termination_start_ab2_id ), ...
          IF (
                  _termination_start_ab2_id IS NOT null
              AND _next_ab2 IS NULL
              AND _turned_around IS false
          ) THEN
              -- ... dann merken wir uns dies Richtungsänderung ...
              _turned_around := true;
              -- ... und setzen den aktuellen AG erst einmal auf den Mittelpunkt-AG.
              _current_ab2 := ab2 FROM ab2 WHERE a2_id = _termination_start_ab2_id;

              -- Eigentliches Ändern der Richtung.
              IF ( _direction = 'forward' ) THEN
                  _direction := 'backward';
              ELSE
                  _direction := 'forward';
              END IF;

              -- Je nach neuer Richtung, den Nachfolger bzw. den Vorgänger des Mittelpunkt-AG als eigentlichen neuen aktuellen AG setzen.
              IF ( _direction = 'forward' ) THEN
                 -- Neue Richtung: vorwärts -> aktueller AG: Nachfolger des Mittlelpunkt-AG
                  _current_ab2 := ab2 FROM scheduling.vertex
                                      JOIN ab2 ON ab2_a2_id = a2_id
                                     WHERE
                                           a2_n > _current_ab2.a2_n
                                       AND a2_ab_ix = _abk_ix
                                     ORDER BY a2_n ASC LIMIT 1;
              ELSE
                  -- Neue Richtung: rückwärts -> aktueller AG: Vorgänger des Mittlelpunkt-AG
                  _current_ab2 := ab2 FROM scheduling.vertex
                                      JOIN ab2 ON ab2_a2_id = a2_id
                                     WHERE
                                           a2_n < _current_ab2.a2_n
                                       AND a2_ab_ix = _abk_ix
                                     ORDER BY a2_n DESC LIMIT 1;
              END IF;

              -- _current_ab2 := _next_ab2;

              CONTINUE abkloop;
          END IF;

          -- raise notice 'elements in results: %', (select count(*) from scheduling.resource_termination_results where search_id = _search_id);
          -- raise notice '';
          -- push ab2 forward
          _current_ab2 := _next_ab2;

          IF ( _current_ab2 IS NULL ) THEN
            _abk_done := true;
          END IF;

      END LOOP;

      -- TODO was macht das?
      DELETE FROM scheduling.resource_termination_results
      WHERE
          search_id = _search_id
      AND (
               ab2_a2_id_path[1] <> _ab2_first.a2_id
               -- TODO was macht das? Pfad-Längen vergleichen?
            OR ab2_a2_id_path[ array_length( ab2_a2_id_path, 1)] <> _ab2_last.a2_id
          );

      RETURN _search_id;

  END $$ LANGUAGE plpgsql;
--
